// GSAP Advanced Animations & Effects
$(document).ready(function () {
    if (typeof gsap !== "undefined") {
        gsap.registerPlugin(ScrollTrigger, SplitText);

        // 1. Basic Scroll Triggers (FadeIn, ZoomIn)
        gsap.utils.toArray('.tmp-scroll-trigger').forEach(element => {
            const animation = element.dataset.animation || 'fadeIn';
            const duration = parseFloat(element.dataset.duration) || 1;
            const delay = parseFloat(element.dataset.delay) || 0;
            const start = element.dataset.start || 'top 80%';

            let animationObj;
            if (animation === 'fadeIn') {
                animationObj = { opacity: 0, y: 50 };
            } else if (animation === 'zoomIn') {
                animationObj = { opacity: 0, scale: 0.8 };
            } else {
                animationObj = { opacity: 0 };
            }

            gsap.fromTo(element, animationObj, {
                opacity: 1,
                y: 0,
                scale: 1,
                duration: duration,
                delay: delay,
                scrollTrigger: {
                    trigger: element,
                    start: start,
                    toggleActions: 'play none none none'
                }
            });
        });

        // 2. Text Reveal (SplitText - Letter by Letter)
        const isRTL = document.documentElement.dir === 'rtl';
        const revealTextElements = document.querySelectorAll('.text-reveal');
        revealTextElements.forEach(el => {
            const split = new SplitText(el, { type: isRTL ? 'words' : 'chars' });
            gsap.from(split.chars, {
                opacity: 0,
                y: 20,
                stagger: 0.02,
                duration: 0.5,
                ease: "power2.out",
                scrollTrigger: {
                    trigger: el,
                    start: 'top 85%',
                    toggleActions: 'play none none none'
                }
            });
        });

        // 3. SplitText Collab (Line/Word/Char stagger)
        const splitTextElements = document.querySelectorAll('.split-collab');
        splitTextElements.forEach(el => {
            const split = new SplitText(el, { type: isRTL ? 'words' : 'lines,words', linesClass: 'split-line' });
            gsap.from(split.chars, {
                opacity: 0,
                y: 50,
                rotateX: -90,
                stagger: 0.01,
                duration: 0.8,
                scrollTrigger: {
                    trigger: el,
                    start: 'top 80%',
                    toggleActions: 'play none none none'
                }
            });
        });

        // 4. Parallax Backgrounds
        gsap.utils.toArray(".parallax-bg").forEach((bg) => {
            const speed = bg.dataset.speed || 0.5;
            gsap.to(bg, {
                backgroundPositionY: "100%",
                ease: "none",
                scrollTrigger: {
                    trigger: bg,
                    start: "top bottom",
                    end: "bottom top",
                    scrub: true
                }
            });
        });

        // 5. Custom Cursor
        const cursor = document.querySelector(".custom-cursor");
        if (cursor) {
            window.addEventListener("mousemove", (e) => {
                gsap.to(cursor, {
                    x: e.clientX,
                    y: e.clientY,
                    duration: 0.3,
                    ease: "power2.out"
                });
            });

            document.querySelectorAll("a, button, .tmponhover").forEach(el => {
                el.addEventListener("mouseenter", () => cursor.classList.add("active"));
                el.addEventListener("mouseleave", () => cursor.classList.remove("active"));
            });
        }

        // 6. Mouse Parallax
        document.addEventListener('mousemove', (e) => {
            const moveX = (e.clientX - window.innerWidth / 2) / 100;
            const moveY = (e.clientY - window.innerHeight / 2) / 100;

            document.querySelectorAll('.mouse-move').forEach(el => {
                const speed = el.dataset.speed || 1;
                gsap.to(el, {
                    x: moveX * speed * 20,
                    y: moveY * speed * 20,
                    duration: 1,
                    ease: 'power2.out'
                });
            });
        });

        // 7. Hover effect
        document.querySelectorAll('.tmponhover').forEach(card => {
            card.addEventListener('mouseenter', () => {
                gsap.to(card, { y: -10, duration: 0.3, ease: 'power2.out' });
            });
            card.addEventListener('mouseleave', () => {
                gsap.to(card, { y: 0, duration: 0.3, ease: 'power2.out' });
            });
        });
    }
});
